﻿using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using VA.PPMS.Context;

using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Documents;


namespace VA.PPMS.CRM.Plugins
{
    public class ProviderDuplicateDetection : IPlugin
    {
        private ITracingService tracingService;
        private const string PluginName = "ProviderDuplicateDetection";
        private const string MessageCreate = "CREATE";
        private const string MessageUpdate = "UPDATE";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];


                tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                IOrganizationService service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    //This plugin will be triggered by create/updates to different entities. We will handle the logic slightly different depending on which 
                    //message/event triggered the plugin. 

                    switch (entity.LogicalName)
                    {
                        //handle event first based on initiating entity. 
                        case "ppms_provideridentifier":
                            // handle event based on message type
                            string thisProviderIdentifer = entity.GetAttributeValue<string>("ppms_provideridentifier");
                            switch (context.MessageName.ToUpper())
                            {
                                case MessageCreate:
                                    tracingService.Trace("Create of Provider Identifier: Check for Duplicates");
                                    ProvIdentifierCheckForDuplicates(service, entity, thisProviderIdentifer);
                                    break;
                                case MessageUpdate:
                                    tracingService.Trace("Update of Provider Identifier: Check for Duplicates");
                                    Entity preEntity = (Entity)context.PreEntityImages["PreImage"];
                                    ProvIdentifierCheckForDuplicates(service, preEntity, thisProviderIdentifer);
                                    break;                               
                            }
                            break;
                        case "account":
                            // handle event based on message type
                            string thisProvidersProviderIdentifer = entity.GetAttributeValue<string>("ppms_provideridentifier");
                            switch (context.MessageName.ToUpper())
                            {
                                case MessageCreate:
                                    tracingService.Trace("Create of Provider: Check for Duplicates");
                                    ProviderCheckDuplicates(service, entity, thisProvidersProviderIdentifer);
                                    break;
                                case MessageUpdate:
                                    tracingService.Trace("Update of Provider: Check for Duplicates");
                                    Entity preEntity = (Entity)context.PreEntityImages["PreImage"];
                                    ProviderCheckDuplicates(service, preEntity, thisProvidersProviderIdentifer);
                                    break;
                            }
                            break;
                        default:
                            tracingService.Trace("Valid event message/entity not found");
                            break;
                    }
                    
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            tracingService.Trace("Done");
        }     
        private void ProvIdentifierCheckForDuplicates(IOrganizationService service, Entity entity, string thisProviderIdentifer)
        {           
            using (var svc = new PpmsContext(service))
            {
                tracingService.Trace("Checking for a Duplicate Provider Identifier");
                
                OptionSetValue thisProviderIdentifierType = entity.GetAttributeValue<OptionSetValue>("ppms_identifiertype");
                //If this is not an NPI Identifier don't do the duplicate check.
                if(thisProviderIdentifierType.Value != (int)ppms_ProviderIdentifierType.NPI) { return; }

                //Check for an existing Active Provider Identifer with same Identifier
                var matchingProvIdentifier = svc.ppms_provideridentifierSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(thisProviderIdentifer) && i.Id != entity.Id && i.StateCode.Value == ppms_provideridentifierState.Active);
                if (matchingProvIdentifier != null)
                {
                    tracingService.Trace("Duplicate Provider Identifier found");
                    throw new InvalidPluginExecutionException("A Duplicate Provider Identifier already exists in PPMS. Please check the Provider Identifer list to find existing Provider Identifier.    ");                  
                }
                tracingService.Trace("No Duplicate Provider Identifier found");
                return;         
            }
        }

        private void ProviderCheckDuplicates(IOrganizationService service, Entity entity, string thisProvidersProviderIdentifer)
        {
            using (var svc = new PpmsContext(service))
            {
                tracingService.Trace("Checking for a Provider with a Duplicate Provider Identifier");
               
                OptionSetValue thisProvidersProviderIdentifierType = entity.GetAttributeValue<OptionSetValue>("ppms_provideridentifiertype");

                tracingService.Trace("This Provider's Provider Identifier: " + thisProvidersProviderIdentifer);
                
                //If this is not an NPI Identifier don't do the duplicate check.
                if (thisProvidersProviderIdentifierType.Value != (int)ppms_ProviderIdentifierType.NPI) { return; }

                var matchingProvider = svc.AccountSet.FirstOrDefault(i => i.ppms_ProviderIdentifier.Equals(thisProvidersProviderIdentifer) && i.Id != entity.Id && i.StateCode.Value == (int)PpmsHelper.AccountState.Active);
                if (matchingProvider != null)
                {
                    tracingService.Trace("Provider with a Duplicate Provider Identifier found");
                    throw new InvalidPluginExecutionException("A Provider with a matching Provider Identifier already exists in PPMS. Please check the Provider list to find the existing provider with the associated Provider Identifier.       ");
                }
                tracingService.Trace("No Provider with Duplicate Provider Identifier found");
                return;
            }
        }
    }
}
